const cron = require('node-cron');
const Video = require('../models/Video');
const VideoController = require('../controllers/videoController');
const logger = require('../utils/logger');

// Schedule the cron job to run every five minutes
cron.schedule('*/5 * * * *', async () => {
    try {
        logger.info('Checking for pending video generation tasks...');

        // Find all pending videos
        const pendingVideos = await Video.findAll({
            where: { generationStatus: 'pending' },
            limit: 10 // Process a maximum of 10 videos at a time
        });

        if (pendingVideos.length > 0) {
            logger.info(`Found ${pendingVideos.length} pending video(s). Processing...`);

            for (const video of pendingVideos) {
                try {
                    // Update the status to 'processing'
                    await Video.update(
                        { generationStatus: 'processing' },
                        { where: { jobId: video.jobId } }
                    );

                    // Process the video generation
                    await VideoController.processVideoGenerationJob({
                        data: {
                            jobId: video.jobId,
                            videoDetails: {
                                videoType: video.videoType,
                                voiceType: video.voiceType,
                                artStyle: video.artStyle,
                                aspectRatio: video.aspectRatio,
                                duration: video.duration,
                                platform: video.platform,
                                email: video.email,
                                prompt: video.prompt
                            }
                        }
                    });

                    logger.info(`Video generation completed for jobId: ${video.jobId}`);
                } catch (error) {
                    logger.error(`Error processing video generation for jobId ${video.jobId}: ${error.message}`);
                    await Video.update(
                        { generationStatus: 'failed' },
                        { where: { jobId: video.jobId } }
                    );
                }
            }
        } else {
            logger.info('No pending video generation tasks found.');
        }
    } catch (error) {
        logger.error(`Error in cron job: ${error.message}`);
    }
});

logger.info('Cron job scheduled to check for pending video generation tasks every five minutes.');