const axios = require('axios');
const User = require('../models/User');
const jwt = require('jsonwebtoken');
const crypto = require('crypto');


class TikTokAuthService {
    constructor() {
        this.clientKey = process.env.TIKTOK_CLIENT_KEY;
        this.clientSecret = process.env.TIKTOK_CLIENT_SECRET;
        this.redirectUri = process.env.TIKTOK_REDIRECT_URI;
        this.authUrl = 'https://www.tiktok.com/v2/auth/authorize/';
        this.tokenUrl = 'https://www.tiktok.com/v2/auth/token/';
    }

    generateCodeVerifier() {
        const codeVerifier = crypto.randomBytes(32).toString('base64url');
        return codeVerifier;
    }

    async generateCodeChallenge(codeVerifier) {
        const hashedBuffer = await crypto.subtle.digest(
            'SHA-256',
            new TextEncoder().encode(codeVerifier)
        );
        return Buffer.from(hashedBuffer).toString('base64url');
    }

    async getAuthUrl() {
        const codeVerifier = this.generateCodeVerifier();
        const codeChallenge = await this.generateCodeChallenge(codeVerifier);
        const scopes = ['user.info.basic', 'video.upload', 'video.publish'];
        const params = new URLSearchParams({
            client_key: this.clientKey,
            response_type: 'code',
            scope: scopes.join(','),
            redirect_uri: this.redirectUri,
            state: 'stateString',
            code_challenge: codeChallenge,
            code_challenge_method: 'S256'
        });
        return {
            url: `${this.authUrl}?${params.toString()}`,
            codeVerifier
        };
    }

    async handleCallback(code, codeVerifier) {
        try {
            const params = new URLSearchParams({
                client_key: this.clientKey,
                client_secret: this.clientSecret,
                code,
                grant_type: 'authorization_code',
                redirect_uri: this.redirectUri,
                code_verifier: codeVerifier // Include the stored code verifier
            });
            const response = await axios.post(this.tokenUrl, params, {
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded'
                }
            });

            const { access_token, refresh_token, open_id } = response.data.data;

            // Fetch user info (if required)
            const userInfoResponse = await axios.get(
                `https://open.tiktokapis.com/v2/user/info/`,
                {
                    headers: {
                        Authorization: `Bearer ${access_token}`,
                    },
                }
            );
            const tiktokUserInfo = userInfoResponse.data.data;

            // Check if user already exists
            let user = await User.findOne({ where: { email: tiktokUserInfo.email } });

            if (!user) {
                // Create new user
                user = await User.create({
                    email: tiktokUserInfo.email,
                    name: tiktokUserInfo.display_name,
                    tiktokId: open_id,
                    tiktokTokens: {
                        access_token,
                        refresh_token,
                    },
                });
            } else {
                await user.update({
                    tiktokId: open_id,
                    tiktokTokens: {
                        access_token,
                        refresh_token,
                    },
                });
            }

            // Generate JWT token for authentication
            const jwtToken = jwt.sign({ id: user.id }, process.env.JWT_SECRET, {
                expiresIn: process.env.TOKEN_EXPIRE_TIME,
            });

            return jwtToken;
        } catch (error) {
            console.error('Error handling TikTok OAuth callback:', error);
            throw error;
        }
    }

    async refreshAccessToken(refreshToken) {
        try {
            const response = await axios.post(this.tokenUrl, {
                client_key: this.clientKey,
                client_secret: this.clientSecret,
                refresh_token: refreshToken,
                grant_type: 'refresh_token',
            });

            return response.data.data;
        } catch (error) {
            console.error('Error refreshing TikTok access token:', error);
            throw error;
        }
    }
}

module.exports = new TikTokAuthService();
