const ffmpeg = require('fluent-ffmpeg');
const ffmpegPath = require('ffmpeg-static');
const fs = require('fs');
const fsPromises = fs.promises;
const path = require('path');
const { v4: uuidv4 } = require('uuid');
const axios = require('axios');
const logger = require('../utils/logger');
const cloudinary = require('../utils/cloudinaryConfig');

ffmpeg.setFfmpegPath(ffmpegPath);

class ffmpegService {
  async downloadImage(imageUrl) {
    const tempDir = path.join(__dirname, '../temp/images');
    await fsPromises.mkdir(tempDir, { recursive: true });
    const filename = path.join(tempDir, `image_${uuidv4()}.jpeg`);

    try {
      logger.info(`Downloading image: ${imageUrl}`);
      const response = await axios({
        method: 'get',
        url: imageUrl,
        responseType: 'arraybuffer',
        headers: {
          'User-Agent': 'Mozilla/5.0'
        },
      });

      await fsPromises.writeFile(filename, response.data);
      logger.info(`Image downloaded to: ${filename}`);
      return filename;
    } catch (error) {
      logger.error(`Error downloading image ${imageUrl}: ${error}`);
      throw error;
    }
  }

  async downloadAudio(audioURL) {
    const tempDir = path.join(__dirname, '../temp/cloudinary_audio');
    await fsPromises.mkdir(tempDir, { recursive: true });
    const filename = path.join(tempDir, `audio_${uuidv4()}.mp3`);

    try {
      logger.info(`Downloading audio from: ${audioURL}`);
      const response = await axios({
        method: 'get',
        url: audioURL,
        responseType: 'stream',
        headers: {
          'User-Agent': 'Mozilla/5.0'
        },
      });

      const writer = fs.createWriteStream(filename);
      response.data.pipe(writer);

      await new Promise((resolve, reject) => {
        writer.on('finish', () => {
          logger.info(`Audio downloaded to: ${filename}`);
          resolve();
        });
        writer.on('error', reject);
      });

      return filename;
    } catch (error) {
      logger.error(`Error downloading audio ${audioURL}: ${error}`);
      throw error;
    }
  }

  async generateVideo({ imagePaths, audioURL, sentenceTimings, enableCaptions, captionStyle }) {
    logger.info('Starting video generation...');
    if (!Array.isArray(imagePaths) || imagePaths.length === 0) {
      throw new Error('imagePaths must be a non-empty array');
    }

    if (!audioURL) {
      throw new Error('Audio URL is required.');
    }

    try {
      const audioPath = await this.downloadAudio(audioURL);

      const tempVideoDir = path.join(__dirname, '../temp/video');
      await fsPromises.mkdir(tempVideoDir, { recursive: true });

      const rawVideoPath = path.join(tempVideoDir, `video-${uuidv4()}.mp4`);
      const finalVideoPath = path.join(tempVideoDir, `final-video-${uuidv4()}.mp4`);

      logger.info('Downloading images and pairing with durations...');
      const imagePathsWithDurations = [];
      for (let i = 0; i < imagePaths.length; i++) {
        const img = await this.downloadImage(imagePaths[i]);
        const duration = sentenceTimings[i]?.duration || 3;
        logger.info(`Image ${i + 1}: ${img} - duration: ${duration}`);
        imagePathsWithDurations.push({ path: img, duration });
      }

      const imageListPath = path.join(__dirname, '../temp/image_list.txt');
      const imageListContent = imagePathsWithDurations
        .map(image => `file '${image.path}'\nduration ${image.duration}`)
        .join('\n') + `\nfile '${imagePathsWithDurations.at(-1).path}'`;

      await fsPromises.writeFile(imageListPath, imageListContent, { encoding: 'utf8' });
      logger.info(`FFmpeg image list file written to: ${imageListPath}`);

      logger.info('Combining images into raw video...');
      await new Promise((resolve, reject) => {
        ffmpeg()
          .input(imageListPath)
          .inputOptions(['-f concat', '-safe 0'])
          .videoCodec('libx264')
          .outputOptions(['-pix_fmt yuv420p', '-s 1920x1080', '-r 30'])
          .save(rawVideoPath)
          .on('start', cmd => logger.info('FFmpeg raw video generation started: ' + cmd))
          .on('end', () => {
            logger.info('Raw video created successfully.');
            resolve();
          })
          .on('error', err => {
            logger.error('Error during raw video generation:', err);
            reject(err);
          });
      });

      // === Generate Drawtext Filter ===
      logger.info('Generating drawtext filter for captions...');
      let currentTime = 0;
      
      function wrapSentence(sentence, maxWordsPerLine = 5) {
        const words = sentence.split(' ');
        const lines = [];
        for (let i = 0; i < words.length; i += maxWordsPerLine) {
          lines.push(words.slice(i, i + maxWordsPerLine).join(' '));
        }
        return lines;
      }

      // Improved text escaping function for FFmpeg
      function escapeTextForFFmpeg(text) {
        return text
          .replace(/\\/g, '\\\\')    // Escape backslashes first
          .replace(/'/g, "'\\''")    // Escape single quotes properly for FFmpeg
          .replace(/:/g, '\\:')      // Escape colons
          .replace(/\[/g, '\\[')     // Escape square brackets
          .replace(/\]/g, '\\]')
          .replace(/,/g, '\\,')      // Escape commas
          .replace(/;/g, '\\;')      // Escape semicolons
          .replace(/"/g, '\\"');     // Escape double quotes
      }

      // Function to get background color based on text color
      function getBackgroundColor(textColor) {
        return textColor.toLowerCase() === 'white' ? 'black@0.7' : 'white@0.7';
      }

      const drawtextFilter = sentenceTimings.map(({ sentence, duration }) => {
        const start = currentTime;
        const end = currentTime + duration;
        currentTime = end;

        const { fontColor, fontSize, position, font } = captionStyle;
        const lineSpacing = fontSize + 30;
        const lines = wrapSentence(sentence);
        const totalLines = lines.length;
        const backgroundColor = getBackgroundColor(fontColor);

        function getBaseY(position, lineIndex, spacing, total) {
          switch (position) {
            case 'top': return 50 + lineIndex * spacing;
            case 'bottom': return `h - ${(total - lineIndex) * spacing} - 50`;
            default:
              return `(h - ${spacing * total}) / 2 + ${lineIndex * spacing}`;
          }
        }

        return lines.map((line, index) => {
          const escapedLine = escapeTextForFFmpeg(line);
          const y = getBaseY(position, index, lineSpacing, totalLines);
          
          // Create drawtext filter with background box
          return `drawtext=text='${escapedLine}':enable='between(t,${start.toFixed(2)},${end.toFixed(2)})':fontcolor=${fontColor}:fontsize=${fontSize}:font='${font}':x=(w-text_w)/2:y=${y}:box=1:boxcolor=${backgroundColor}:boxborderw=10`;
        }).join(',');
      }).join(',');

      logger.info('Drawtext filter with background overlay generated.');

      logger.info('Combining raw video with audio and subtitles...');
      await new Promise((resolve, reject) => {
        const command = ffmpeg(rawVideoPath)
          .input(audioPath);

        if (enableCaptions && drawtextFilter) {
          command.videoFilter(drawtextFilter);
          logger.info('Drawtext filter with background overlay applied.');
        }

        command
          .outputOptions(['-c:a', 'aac', '-b:a', '192k'])
          .save(finalVideoPath)
          .on('start', cmd => logger.info('FFmpeg final video processing started: ' + cmd))
          .on('end', () => {
            logger.info('Final video created successfully.');
            resolve();
          })
          .on('error', err => {
            logger.error('Error during final video generation:', err);
            reject(err);
          });
      });

      logger.info('Uploading final video to Cloudinary...');
      const uploadResult = await cloudinary.uploader.upload(finalVideoPath, {
        resource_type: 'video',
        folder: 'generated_videos'
      });

      if (!uploadResult.secure_url) {
        throw new Error('Cloudinary upload failed.');
      }

      logger.info(`Video uploaded to Cloudinary: ${uploadResult.secure_url}`);

      logger.info('Cleaning up temporary files...');
      await Promise.allSettled([
        fsPromises.unlink(imageListPath),
        fsPromises.unlink(rawVideoPath),
        fsPromises.unlink(finalVideoPath),
        fsPromises.unlink(audioPath),
        ...imagePathsWithDurations.map(img => fsPromises.unlink(img.path))
      ]);
      logger.info('Cleanup complete.');

      return uploadResult.secure_url;
    } catch (err) {
      logger.error('Error in generateVideo:', err);
      throw err;
    }
  }
}

module.exports = new ffmpegService();