const OpenAI = require('openai');
const logger = require('../utils/logger');
require('dotenv').config();

class OpenAIService {
  constructor() {
    if (!process.env.OPENAI_API_KEY) {
      throw new Error('Missing OpenAI API Key');
    }
    this.client = new OpenAI({
      apiKey: process.env.OPENAI_API_KEY, // Default configuration
    });
  }

  async refinePrompt(voiceoverScript, style) {
    if (!voiceoverScript || typeof voiceoverScript !== 'string') {
      throw new Error('Invalid voiceover script');
    }
    if (!style || typeof style !== 'string') {
      throw new Error('Invalid art style');
    }

    try {
      const response = await this.client.chat.completions.create({
        model: 'gpt-4',
        messages: [
          {
            role: 'system',
            content: `You are an expert at creating detailed image prompts that align closely with spoken content. 
            The goal is to generate a single image prompt that encapsulates the essence of the following voiceover script. 
            Make sure the prompt is clear, detailed, and descriptive of the key visual elements in the script. IT SHOULD NOT BE MORE THAN 1400 CHARACTERS.`,
          },
          {
            role: 'user',
            content: `Refine the following voiceover script into a detailed image prompt for Leonardo AI:

            Script: "${voiceoverScript}"

            Ensure that the visual elements match the narrative in the script. Avoid text in the image. Provide a clear, descriptive scene that corresponds to the tone and content of the voiceover. Make sure the prompt length is less than 1400 CHARACTERS. Note that it's generating multiple images.`,
          },
        ],
      });

      return response.choices[0].message.content.trim();
    } catch (err) {
      logger.error(`OpenAI refinement error: ${err.message}`);
      throw new Error('Error Refining Prompt');
    }
  }

  async generateVoiceoverScript(genre, duration, character) {
    if (!genre || typeof genre !== 'string') {
        throw new Error('Invalid genre');
    }
    if (!duration || typeof duration !== 'number') {
        throw new Error('Invalid duration');
    }

    const charLimit = character - 100;
    const maxTokens = Math.floor(charLimit / 4); // Approximate token conversion

    try {
        const response = await this.client.chat.completions.create({
            model: 'gpt-4',
            max_tokens: maxTokens,  // Enforce approximate limit
            messages: [
                {
                    role: 'system',
                    content: `You are a professional scriptwriter and voiceover expert.

                      CRITICAL REQUIREMENTS:
                      1) Return ONLY the spoken narration for a voiceover script.
                      2) DO NOT include any scene descriptions, camera directions, or screenplay-style formatting.
                      3) The script should flow naturally as if spoken by a voiceover artist.
                      4) The narration must NOT exceed ${charLimit} characters.
                      5) Do NOT include any quotation marks (like " or '), apostrophes, or special characters such as !, ?, :, -, /, etc.
                      6) Ensure that sentences are complete. Do NOT cut sentences in half.
                      7) Use simple, clear sentences — no poetic formatting or unusual punctuation.`,
                },
                {
                    role: 'user',
                    content: `Create a voiceover script for a ${genre} video that lasts ${
                      duration - 5
                    } seconds. STRICT LIMIT: The script must be ${charLimit} characters or fewer. 
                    Always complete full sentences, and never exceed the character limit.`,
                },
            ],
        });

        let script = response.choices[0].message.content.trim();

        // Ensure the script does not exceed charLimit while keeping full sentences
        if (script.length > charLimit) {
            const trimmedScript = script.slice(0, charLimit);
            const lastFullStopIndex = trimmedScript.lastIndexOf('.');

            if (lastFullStopIndex !== -1) {
                script = trimmedScript.slice(0, lastFullStopIndex + 1); // Keep full sentence
            }
        }

        return script;
    } catch (err) {
        logger.error(`Voiceover script generation error: ${err.message}`);
        throw new Error('Error generating voiceover script');
    }
}

  

  async generateImages(imagePrompt, duration, aspectRatio) {
    if (!imagePrompt || typeof imagePrompt !== 'string') {
      throw new Error('Invalid image prompt: must be a non-empty string');
    }
    if (!duration || typeof duration !== 'number') {
      throw new Error('Invalid duration: must be a number');
    }

    const allowedAspectRatios = ['256x256', '512x512', '1dav24x1024'];
    if (!allowedAspectRatios.includes(aspectRatio)) {
      throw new Error(`Invalid aspect ratio: must be one of ${allowedAspectRatios.join(', ')}`);
    }

    try {
      const response = await this.client.images.create({
        prompt: imagePrompt,
        n: Math.max(1, Math.floor(duration / 4)),
        size: aspectRatio,
      });

      return response.data.map((image) => image.url);
    } catch (err) {
      logger.error(`Image generation error: ${err.message}`);
      throw err.message;
    }
  }

  async generateVideoDetails(userPrompt) {
    if (!userPrompt || typeof userPrompt !== 'string') {
      throw new Error('Invalid user prompt: must be a non-empty string in openAI');
    }

    try {
      const response = await this.client.chat.completions.create({
        model: 'gpt-4',
        messages: [
          {
            role: 'system',
            content: `You are an expert content creator specializing in YouTube metadata. Generate a title, description, and tags for a YouTube video based on the following details.`,
          },
          {
            role: 'user',
            content: `Video details:
          Prompt: "${userPrompt}"
          
          Provide the response as a JSON object with the following structure:
          {
            "title": "string",
            "description": "string",
            "tags": ["string"]
          }`,
          },
        ],
      });

      return JSON.parse(response.choices[0].message.content.trim());
    } catch (err) {
      logger.error(`Video details generation error: ${err.message}`);
      throw new Error('Error generating youtube video details.');
    }
  }

  async generateCustomPrompt() {
    const duration = 0;  
    try {
      const response = await this.client.chat.completions.create({
        model: 'gpt-4',
        messages: [
          {
            role: 'system',
            content: `You are a professional scriptwriter and voiceover expert writing scripts for a video generation applications using Eleven Labs for audio, Leonardo for images, and FFmpeg for video synthesis. CRITICAL REQUIREMENTS:
            1) Return ONLY voice over script text.
            2) DO NOT use placeholders like [Company Name] or brackets.
            3) The script should flow naturally as if narrated by a human.
            4) Maintain genre-appropriate tone and style.
            5) Use vivid and engaging descriptions for scenes and themes.
            6) Ensure the prompt is adaptable to various video genres.
            7) Maintain compatibility with image generation models like Leonardo.
            8) Ensure the prompt is concise and well-structured for a ${duration}-second video, fitting approximately 2.5–3 words per second.`,
          },
          {
            role: 'user',
            content: `Generate a default creative prompt for a ${duration}-second visually stunning and engaging video based on a random interesting. Focus on:
            - Vivid descriptions of the scenes and themes.
            - Genre adaptability.
            - Compatibility with image and video synthesis tools.`,
          },
        ],
      });
  
      return response.choices[0].message.content.trim();
    } catch (err) {
      logger.error(`Custom prompt generation error for theme "${theme}" and duration ${duration}: ${err.message}`);
      throw new Error('Error generating custom prompt.');
    }
  }

  async moderateText(text) {
    if (!text || typeof text !== 'string') {
      throw new Error('Invalid input: text must be a non-empty string');
    }
  
    try {
      const response = await this.client.moderations.create({
        input: text,
      });

      const flagged = response.results[0].flagged;
  const categoryScores = response.results[0].category_scores;

  const threshold = 0.001;  
  const highScores = Object.entries(categoryScores).filter(([category, score]) => score >= threshold);

  // If any category has a score >= threshold, it's "Not Moderate"
  const isNotModerate = highScores.length > 0;
      return {
        flagged,
        highScores,
        isNotModerate,
      };
    } catch (err) {
      logger.error(`Text moderation error: ${err.message}`);
      throw new Error('Error processing text moderation');
    }
  }
}

module.exports = new OpenAIService();
