const axios = require("axios");
const logger = require("../utils/logger");
require("dotenv").config();

class LeonardoService {
    constructor() {
        if (!process.env.LEONARDO_API_KEY) {
            throw new Error("LEONARDO_API_KEY is not set");
        }

        this.apiKey = process.env.LEONARDO_API_KEY;
        this.baseUrl = "https://cloud.leonardo.ai/api/rest/v1";
    }

    async generateImages(imagePrompt, artStyle, aspectRatio, duration) {
        if (!imagePrompt || typeof imagePrompt !== "string") {
            throw new Error("Invalid image prompt: must be a non-empty string");
        }

        if (!duration || typeof duration !== "number") {
            throw new Error("Invalid duration: must be a number");
        }

        if (!aspectRatio || typeof aspectRatio !== "string") {
            throw new Error("Invalid aspect ratio: must be a string in 'widthxheight' format");
        }

        try {
            const [width, height] = aspectRatio.split("x").map(Number);

            if (isNaN(width) || isNaN(height)) {
                throw new Error("Invalid aspect ratio values");
            }

            const url = `${this.baseUrl}/generations`;
            const headers = {
                accept: "application/json",
                authorization: `Bearer ${this.apiKey}`,
                "content-type": "application/json",
            };
            const truncatedPrompt = imagePrompt.length > 1500 ? imagePrompt.substring(0, 1500) : imagePrompt;

            const data = {
                prompt: truncatedPrompt,
                modelId: process.env.LEONARDO_MODEL_ID,
                contrast: 3,
                styleUUID: artStyle,
                width: width,
                height: height,
                num_images: Math.min(8, Math.max(1, Math.floor(duration / 4))), // an image per four seconds.
                enhancePrompt: false, // Optional: Set to true if you want to enhance the prompt
            };

            const response = await axios.post(url, data, { headers });

            return response.data.sdGenerationJob.generationId;
        } catch (err) {
            console.log(err)
            logger.error(`Error during image generation: ${err.response?.data.error || err.message}`);
            throw new Error (
                err.response.data.error || "Error getting image generation Id"
            )
        }
    }

    async getGeneratedImages(generationId) {
        try {
            const url = `${this.baseUrl}/generations/${generationId}`;
            const headers = {
                accept: "application/json",
                authorization: `Bearer ${this.apiKey}`,
            };

            const response = await axios.get(url, { headers });

            const images = response.data.generations_by_pk.generated_images;
            const imageUrls = images.map(image => image.url);
            return imageUrls;
        } catch (err) {
            logger.error(err);
            logger.error(`Error fetching generated images: ${err.response?.data.error || err.message}`);
            throw new Error (
                err.response.data.error || "Error fetching generated images"
            )

        }
    }
}

module.exports = new LeonardoService();