const express = require('express');
const router = express.Router();
const VideoController = require('../controllers/videoController');
const fetchDataController = require('../controllers/fetchDataController');
const { apiKeyMiddleware, authMiddleware } = require('../middlewares/authMiddleware');

// Custom middleware to conditionally apply either authMiddleware or apiKeyMiddleware
const applyMiddleware = (req, res, next) => {
  if (req.header('x-api-key')) {
    // If x-api-key is present, apply only the API key validation middleware
    return apiKeyMiddleware(req, res, next);
  }
  
  // If no x-api-key, apply the authMiddleware
  return authMiddleware(req, res, next);
};

router.post('/generate-initial', applyMiddleware, VideoController.generateInitial);
router.post('/run-wait-time', applyMiddleware, VideoController.waitTimePromise);
router.post('/get-generated-images', applyMiddleware, VideoController.getGeneratedImages);
router.post('/generate-video', applyMiddleware, VideoController.generateVideo);
router.get('/user-progress', applyMiddleware, VideoController.getUserProgress);

router.post('/moderate-text', applyMiddleware, VideoController.moderateText);

router.post('/video/regenerate/:jobId', applyMiddleware, VideoController.regenerateVideo);

router.post('/generate-custom-prompt', applyMiddleware, VideoController.generateCustomPromptBasedOnAnIdea);

router.get('/generate-sample-prompt', applyMiddleware, VideoController.generateCustomPrompt);

router.get('/voices', fetchDataController.getVoices);
// 
router.get('/art-styles', fetchDataController.getArtStyles);

router.get('/videos', applyMiddleware, fetchDataController.getUserVideos);

router.delete('/video/:id', applyMiddleware, fetchDataController.deleteVideo);

router.post('/mark-done', applyMiddleware, VideoController.markJobAsDone);

module.exports = router;
