const bcrypt = require('bcryptjs');
const jwt = require('jsonwebtoken');
const User = require('../models/User');
const { sendAuthEmail } = require('../services/mailService');

// Function to generate reset token
const generateResetToken = (user) => {
  return jwt.sign({ email: user.email, id: user.id }, process.env.JWT_SECRET, { expiresIn: process.env.TOKEN_EXPIRE_TIME });
};

// Function to handle password reset request
const handlePasswordResetRequest = async (email) => {
  const user = await User.findOne({ where: { email } });
  if (!user) {
    throw new Error('User with this email not found');
  }

  const resetToken = generateResetToken(user);

  const resetLink = `${process.env.FRONTEND_URL}/reset-password?token=${resetToken}`;

  // Use the dedicated mail service to send the reset link
  const subject = 'Password Reset Request';
  const html = `<p>Click <a href="${resetLink}">here</a> to reset your password.</p>`;
  await sendAuthEmail(email, subject, html);

};

// Function to reset user password
const resetUserPassword = async (token, password, passwordConfirmation) => {
  if (password !== passwordConfirmation) {
    throw new Error('Passwords do not match');
  }

  const decoded = jwt.verify(token, process.env.JWT_SECRET);
  const user = await User.findOne({ where: { id: decoded.id, email: decoded.email } });

  if (!user) {
    throw new Error('User not found or invalid token');
  }

  const hashedPassword = bcrypt.hashSync(password, 10);
  user.password = hashedPassword;
  await user.save();
};

module.exports = {
  handlePasswordResetRequest,
  resetUserPassword,
};
