const { google } = require('googleapis');
const fs = require('fs');
const path = require('path');
const axios = require('axios');
const logger = require('../utils/logger');

class YouTubeService {
    constructor() {
        this.oauth2Client = new google.auth.OAuth2(
            process.env.GOOGLE_CLIENT_ID,
            process.env.GOOGLE_CLIENT_SECRET,
            process.env.GOOGLE_REDIRECT_URI
        );
    }

    async downloadVideo(videoUrl) {
        try {
            const downloadsDir = path.join(__dirname, '../downloads');
            
            if (!fs.existsSync(downloadsDir)) {
                fs.mkdirSync(downloadsDir, { recursive: true }); 
            }

            const videoPath = path.join(downloadsDir, `video_${Date.now()}.mp4`);
            const writer = fs.createWriteStream(videoPath);

            const response = await axios({
                url: videoUrl,
                method: 'GET',
                responseType: 'stream',
            });

            response.data.pipe(writer);

            return new Promise((resolve, reject) => {
                writer.on('finish', () => resolve(videoPath));
                writer.on('error', (err) => {
                    logger.error(`Failed to write video file: ${err.message}`);
                    reject(new Error(`File write error: ${err.message}`));
                });
            });
        } catch (error) {
            logger.error(`Failed to download video: ${error.message}`);
            throw new Error(`Video download failed: ${error.message}`);
        }
    }

    // Function to upload the downloaded video
    async uploadVideo(refreshToken, videoUrl, videoDetails) {
        try {
            const videoPath = await this.downloadVideo(videoUrl);
            this.oauth2Client.setCredentials({
                refresh_token: refreshToken,
            });
            const { token } = await this.oauth2Client.getAccessToken();
        if (!token) {
            logger.error("Failed to retrieve a new access token.");
        }

        this.oauth2Client.setCredentials({
            access_token: token,
            refresh_token: refreshToken, 
        });

            const youtube = google.youtube({ version: 'v3', auth: this.oauth2Client });

            const response = await youtube.videos.insert({
                part: 'snippet,status',
                requestBody: {
                    snippet: {
                        title: videoDetails.title || 'Generated Video',
                        description: videoDetails.description || 'This video was generated by TubeCentric Video Generator..',
                        tags: videoDetails.tags || ['generated', 'video'],
                    },
                    status: {
                        privacyStatus: videoDetails.privacyStatus || 'unlisted',
                    },
                },
                media: {
                    body: fs.createReadStream(videoPath), // Use the downloaded video file
                },
            });

            logger.info(`Video uploaded to YouTube: ${response.data.id}`);

            // Delete the video after uploading
            fs.unlinkSync(videoPath);

            return response.data.id;
        } catch (error) {
            logger.error(`Failed to upload video to YouTube: ${error.message}`);
            throw new Error(`YouTube upload failed: ${error.message}`);
        }
    }
}

module.exports = new YouTubeService();
