const { createClient } = require('pexels');
const logger = require('../utils/logger');
require('dotenv').config();

class PexelsService {
  constructor() {
    if (!process.env.PEXELS_API_KEY) {
      throw new Error('Missing Pexels API Key');
    }
    this.client = createClient(process.env.PEXELS_API_KEY);
  }

  async fetchImages(query, perPage, orientation) {
    const numberOfImages = Math.max(1, Math.floor(perPage / 4))
    if (!Number.isInteger(numberOfImages) || numberOfImages <= 0) {
      throw new Error('Invalid numberOfImages value. Must be a positive integer.');
    }
    const validOrientations = ['landscape', 'portrait', 'square'];
    if (orientation && !validOrientations.includes(orientation)) {
      throw new Error('Invalid orientation. Must be "landscape", "portrait", or "square".');
    }

    const options = { query, per_page: numberOfImages };
    if (orientation) options.orientation = orientation;

    try {
      const response = await this.client.photos.search(options);

      if (!response || !response.photos || response.photos.length === 0) {
        throw new Error('No images found for the given query.');
      }

      const imageUrls = response.photos.map((photo) => photo.src.large);
      return imageUrls;
    } catch (err) {
      logger.error(`Pexels API fetch error: ${err.message}`);
      throw new Error(`Failed to fetch images: ${err.message}`);
    }
  }
}

module.exports = new PexelsService();
