const { google } = require('googleapis');
const User = require('../models/User');
const jwt = require('jsonwebtoken');

class GoogleAuthService {
    constructor() {
        this.oauth2Client = new google.auth.OAuth2(
            process.env.GOOGLE_CLIENT_ID,
            process.env.GOOGLE_CLIENT_SECRET,
            process.env.GOOGLE_REDIRECT_URI
        );
    }

    getAuthUrl() {
        const scopes = [
            'https://www.googleapis.com/auth/youtube.upload',
            'https://www.googleapis.com/auth/userinfo.email',
            'https://www.googleapis.com/auth/userinfo.profile',
        ];
        return this.oauth2Client.generateAuthUrl({
            access_type: 'offline',
            prompt: 'consent',
            scope: scopes,
        });
    }

    async handleCallback(code) {
        try {
            const { tokens } = await this.oauth2Client.getToken(code);
            this.oauth2Client.setCredentials(tokens);

            // Get user info from Google
            const oauth2 = google.oauth2({ auth: this.oauth2Client, version: 'v2' });
            const { data: googleUserInfo } = await oauth2.userinfo.get();

            // Check if user already exists
            let user = await User.findOne({ where: { email: googleUserInfo.email } });

            if (!user) {
                // Create new user
                throw new Error(`User not found`);

            } else {
                await user.update({
                    googleId: googleUserInfo.id,
                    googleTokens: {
                        access_token: tokens.access_token,
                        refresh_token: tokens.refresh_token,
                        scope: tokens.scope,
                        expiry_date: tokens.expiry_date,
                    },
                    emailVerified: true,
                });
            }

            // Generate JWT token for authentication
            const jwtToken = jwt.sign({ id: user.id }, process.env.JWT_SECRET, { expiresIn: process.env.TOKEN_EXPIRE_TIME });

            return jwtToken;
        } catch (error) {
            console.error('Error handling Google OAuth callback:', error);
            throw error;
        }
    }

    async refreshAccessToken(refreshToken) {
        try {
            this.oauth2Client.setCredentials({
                refresh_token: refreshToken
            });
            const { credentials } = await this.oauth2Client.refreshAccessToken();
            return credentials;
        } catch (error) {
            console.error('Error refreshing access token:', error);
            throw error;
        }
    }
}

module.exports = new GoogleAuthService();