const axios = require("axios");
const fs = require("fs");
const path = require("path");
const { v4: uuidv4 } = require("uuid");
const logger = require("../utils/logger");
const cloudinary = require('../utils/cloudinaryConfig');
require("dotenv").config();

class ElevenLabsService {
  constructor() {
    if (!process.env.ELEVENLABS_API_KEY) {
      throw new Error("ELEVENLABS_API_KEY is not set");
    }

    this.apiKey = process.env.ELEVENLABS_API_KEY;
    this.baseUrl = "https://api.elevenlabs.io/v1";
  }

  async generateAudio(text, voiceType) {
    if (!text || typeof text !== "string") {
      throw new Error("Invalid text input");
    }

    if (!voiceType || typeof voiceType !== "string") {
      throw new Error("Invalid voice type");
    }

    try {
      const url = `${this.baseUrl}/text-to-speech/${voiceType}/with-timestamps?output_format=mp3_44100_128`;
      const headers = {
        "xi-api-key": this.apiKey,
        "Content-Type": "application/json",
      };

      const data = {
        text,
        model_id: "eleven_multilingual_v2",
      };

      // Make the API request using axios
      const response = await axios.post(url, data, {
        headers,
      });

      const alignment = response.data.alignment || null;
      // Prepare the temp directory for saving audio
      const tempDir = path.join(__dirname, "../temp/audio");
      if (!fs.existsSync(tempDir)) {
        fs.mkdirSync(tempDir, { recursive: true });
      }

      // Save the audio to a file
      const audioPath = path.join(tempDir, `audio-${uuidv4()}.mp3`);
      const audioBuffer = Buffer.from(response.data.audio_base64, "base64"); // Extract audio from response
      fs.writeFileSync(audioPath, audioBuffer);
    

      try {
        const uploadResult = await cloudinary.uploader.upload(audioPath, {
          resource_type: 'video',
          folder: 'elevenlabs_audio',
        });

        if (!uploadResult || !uploadResult.secure_url) {
          throw new Error('Cloudinary upload failed: No secure URL returned.');
        }
        fs.unlinkSync(audioPath);
        
        const audioURL = uploadResult.secure_url;
        return {
          audioURL,
          alignment
        };
      } catch (error) {
        logger.error(`Error during Cloudinary upload: ${error}`);

        if (fs.existsSync(audioPath)) {
          fs.unlinkSync(audioPath);
        }
        throw new Error(`Cloudinary upload failed: ${error.message || 'Unknown error'}`);
      }


    } catch (error) {
      logger.error(`Error during text-to-speech conversion: ${error.message}`);
      if (error.response) {
        // Log the API response error details
        logger.error(`API Error: ${JSON.stringify(error.response.data)}`);
        throw new Error(
          error.response.data?.detail?.message || "Failed to generate audio."
        );
      } else {
        throw new Error("Network error: Failed to connect to ElevenLabs API.");
      }
    }
  }
}

module.exports = new ElevenLabsService();