require('dotenv').config();
const jwt = require('jsonwebtoken');
const nodemailer = require('nodemailer');
const { sendAuthEmail } = require('../services/mailService');
const User = require('../models/User');

// Function to send email using Nodemailer
exports.sendVerificationEmail = async (email, token) => {

  const user = await User.findOne({ where: { email: email } });
  if (!user) {
    return res.status(404).json({ error: 'User not found' });
  }
  user.verificationToken = token;
  user.TokenExpiresIn = new Date(Date.now() + 60 * 60 * 1000).toISOString();
  await user.save();
  const verificationUrl = `${process.env.FRONTEND_URL}/verify-email?token=${token}`;

  const emailContent = `<p>Click <a href="${verificationUrl}">here</a> to verify your email.</p>`;

  await sendAuthEmail(email, 'Email Verification', emailContent);
}

// Resend Verification Email
exports.resendVerificationEmail = async (req, res) => {
  try {
    const user = await User.findOne({ where: { email: req.body.email } });
    if (!user) {
      return res.status(404).json({ error: 'User not found' });
    }

    // Generate a token for email verification
    const token = jwt.sign({ email: user.email }, process.env.JWT_SECRET, { expiresIn: process.env.TOKEN_EXPIRE_TIME });

    // Create verification URL and email content
    const verificationUrl = `${process.env.FRONTEND_URL}/verify-email?token=${token}`;
    const emailContent = `<p>Click <a href="${verificationUrl}">here</a> to verify your email.</p>`;

    user.verificationToken = token;
    user.TokenExpiresIn = new Date(Date.now() + 60 * 60 * 1000).toISOString();
    await user.save();

    // Use the mail service to send the email
    await sendAuthEmail(user.email, 'Email Verification', emailContent);

    res.json({ message: 'Verification link sent' });
  } catch (error) {
    res.status(500).json({ error: 'Failed to resend verification link' });
  }
};


// Verify Email
exports.verifyEmail = async (req, res) => {
  const { token } = req.params;
  try {
    const decoded = jwt.verify(token, process.env.JWT_SECRET);
    const user = await User.findOne({ where: { email: decoded.email } });

    if (!user) {
      return res.status(404).json({ error: 'User not found' });
    }

    if (new Date() > new Date(user.TokenExpiresIn)) {
      return res.status(400).json({ error: 'Token has expired' });
    }

    if (user.verificationToken !== token) {
      return res.status(400).json({ error: 'Invalid verification token' });
    }

    if (user.emailVerified) {
      return res.json({ message: 'Email already verified' });
    }

    user.emailVerified = true;
    await user.save();

    res.json({ message: 'Email verified successfully' });
  } catch (error) {
    console.log(error)
    res.status(400).json({ error: 'Invalid or expired token' });
  }
};